<?php

namespace App\Http\Controllers;

use App\Models\Gift;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class GiftController extends Controller
{
    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            // تحقق من صحة البيانات
            $request->validate([
                'title' => 'required|string',
                'category' => 'required|string',
                'imgSrc' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048', // تحقق من صحة صورة imgSrc
                'giftCards.*' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048', // تحقق من صحة الصور
            ]);

            $gift = new Gift();
            $gift->title = $request->input('title');
            $gift->category = $request->input('category');

            // التعامل مع تحميل الصورة في حقل imgSrc
            if ($request->hasFile('imgSrc')) {
                $file = $request->file('imgSrc');
                $path = $file->store('public/img-src'); // تخزين الصورة
                $gift->imgSrc = Storage::url($path); // الحصول على رابط الصورة
            }

            // التعامل مع تحميل الصور في حقل giftCards
            if ($request->hasFile('giftCards')) {
                $files = $request->file('giftCards');
                $giftCards = [];

                foreach ($files as $file) {
                    $path = $file->store('public/gift-cards'); // تخزين الصورة
                    $giftCards[] = [
                        'id' => uniqid(), // تعيين معرف فريد لكل بطاقة هدية
                        'imgSrc' => Storage::url($path) // الحصول على رابط الصورة
                    ];
                }

                $gift->giftCards = $giftCards; // تعيين بيانات بطاقات الهدايا
            }

            $gift->save();

            return response()->json([
                'success' => true,
                'message' => 'Gift created successfully.',
                'data' => $gift
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while creating the gift.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $gifts = Gift::all();
        return response()->json(['data' => $gifts], 200);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $gift = Gift::find($id);

        if (!$gift) {
            return response()->json(['error' => 'Gift not found.'], 404);
        }

        return response()->json(['data' => $gift], 200);
    }

    /**
     * Update the specified resource in storage.
     */
 public function update(Request $request, $id)
{
    try {
        // التحقق من صحة البيانات
        $request->validate([
            'title' => 'nullable|string',
            'category' => 'nullable|string',
            'imgSrc' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
            'giftCards.*' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
        ]);

        $gift = Gift::find($id);

        if (!$gift) {
            return response()->json(['error' => 'Gift not found.'], 404);
        }

        // تحديث الحقول النصية إذا وجدت
        if ($request->has('title')) {
            $gift->title = $request->input('title');
        }

        if ($request->has('category')) {
            $gift->category = $request->input('category');
        }

        // التعامل مع تحميل الصورة في حقل imgSrc
        if ($request->hasFile('imgSrc')) {
            $file = $request->file('imgSrc');
            $path = $file->store('public/img-src');
            $gift->imgSrc = Storage::url($path);
        }

        // التعامل مع تحميل الصور في حقل giftCards
        if ($request->hasFile('giftCards')) {
            $files = $request->file('giftCards');
            $newGiftCards = [];

            foreach ($files as $file) {
                $path = $file->store('public/gift-cards');
                $newGiftCards[] = [
                    'id' => uniqid(),
                    'imgSrc' => Storage::url($path)
                ];
            }

            // فقط إضافة الصور الجديدة إلى giftCards
            if (is_array($gift->giftCards)) {
                $gift->giftCards = array_merge($gift->giftCards, $newGiftCards);
            } else {
                $gift->giftCards = $newGiftCards;
            }
        }

        $gift->save();

        return response()->json(['success' => 'Gift updated successfully.', 'data' => $gift], 200);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'An error occurred while updating the gift.',
            'error' => $e->getMessage()
        ], 500);
    }
}

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $gift = Gift::find($id);

        if (!$gift) {
            return response()->json(['error' => 'Gift not found.'], 404);
        }

        $gift->delete();

        return response()->json(['success' => 'Gift deleted successfully.'], 200);
    }
}

