<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use App\Models\Image;

class ImageController extends Controller
{
    /**
     * Display a listing of the images.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $images = Image::all();
        return response()->json($images);
    }

    /**
     * Store a newly created image in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'num' => 'required|integer',
            'image_url' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($request->hasFile('image_url')) {
            $image = $request->file('image_url');
            $imageName = time() . '.' . $image->getClientOriginalExtension();
            $imagePath = $image->storeAs('images', $imageName, 'public');
            $imgSrc = Storage::url($imagePath); // الحصول على URL الصورة الكامل

            $post = new Image;
            $post->title = $request->title;
            $post->num = $request->num; // تعيين حقل num
            $post->image_url = $imgSrc;
            $post->save();

            return response()->json(['success' => 'Post created successfully.', 'data' => $post], 201);
        } else {
            return response()->json(['error' => 'Failed to upload image.'], 400);
        }
    }

    /**
     * Display the specified image.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $image = Image::findOrFail($id);
        return response()->json($image);
    }

    /**
     * Update the specified image in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */


public function update(Request $request, $id)
{
    // التحقق من صحة البيانات المدخلة
    $validatedData = $request->validate([
        'title' => 'sometimes|required|string|max:255',
        'num' => 'sometimes|required|integer',
        'image_url' => 'sometimes|image|mimes:jpeg,png,jpg,gif|max:2048',
    ], [
        'title.required' => 'يجب إدخال عنوان للصورة.',
        'title.string' => 'العنوان يجب أن يكون نصاً فقط.',
        'title.max' => 'العنوان لا يمكن أن يزيد عن 255 حرفاً.',
        'num.required' => 'يجب إدخال رقم.',
        'num.integer' => 'الرقم يجب أن يكون عدداً صحيحاً.',
        'image_url.image' => 'الملف المرفق يجب أن يكون صورة.',
        'image_url.mimes' => 'نوع الصورة يجب أن يكون: jpeg, png, jpg, gif.',
        'image_url.max' => 'حجم الصورة يجب ألا يتجاوز 2 ميجابايت.',
    ]);

    try {
        // التحقق من وجود الـ id في قاعدة البيانات
        $image = Image::findOrFail($id);

        // التحقق من وجود صورة جديدة
        if ($request->hasFile('image_url')) {
            // حذف الصورة القديمة إذا كانت موجودة
            if ($image->image_url) {
                // تأكد من أن الصورة القديمة موجودة في المسار الصحيح قبل حذفها
                $oldImagePath = 'public/' . $image->image_url; // إضافة public عند حذف الصورة
                if (Storage::exists($oldImagePath)) {
                    Storage::delete($oldImagePath);
                }
            }

            // حفظ الصورة الجديدة
            $newImage = $request->file('image_url');
            $newImageName = time() . '.' . $newImage->getClientOriginalExtension();
            $newImagePath = $newImage->storeAs('public/images', $newImageName);

            // تحديث المسار للصورة في قاعدة البيانات
            $image->image_url = '/storage/images/' . $newImageName; // حفظ المسار نسبياً في قاعدة البيانات
        }

        // تحديث البيانات الأخرى إذا تم إرسالها
        if ($request->filled('title')) {
            $image->title = $request->title;
        }

        if ($request->filled('num')) {
            $image->num = $request->num;
        }

        // حفظ التعديلات في قاعدة البيانات
        $image->save();

        // إرجاع استجابة بنجاح مع البيانات المحدثة ورابط الصورة
        return response()->json([
            'message' => 'تم تحديث الصورة بنجاح.',
            'data' => [
                'id' => $image->id,
                'title' => $image->title,
                'num' => $image->num,
                'image_url' => '/storage/' . $image->image_url, // إضافة /storage فقط مرة واحدة
                'created_at' => $image->created_at,
                'updated_at' => $image->updated_at,
            ]
        ], 200);
    } catch (\Exception $e) {
        // إرجاع استجابة في حال حدوث خطأ
        return response()->json([
            'error' => 'فشل تحديث الصورة.',
            'message' => $e->getMessage(),
        ], 500);
    }
}







    /**
     * Remove the specified image from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
   public function destroy($id)
{
    $image = Image::findOrFail($id);

    // حذف الصورة من التخزين
    if ($image->image_url) {
        $imagePath = str_replace('/storage', 'public', $image->image_url);
        Storage::delete($imagePath);
    }

    $image->delete();

    return response()->json([
        'message' => 'تم حذف الصورة بنجاح.'
    ], 200);
}

}
