<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\MetalPrice;
use Carbon\Carbon;
use Illuminate\Support\Facades\Http; // لإجراء طلبات HTTP
use Illuminate\Support\Facades\Storage;
class MetalPriceController extends Controller
{
    public function getPrices()
    {
        $gold = MetalPrice::where('metal_type', 'gold')->latest('update_date')->first();
        $silver = MetalPrice::where('metal_type', 'silver')->latest('update_date')->first();

        $data = [
            'gold' => $gold ? [
                'goldPriceUpdate' => $gold->price,
                'updateDate' => Carbon::parse($gold->update_date)->toDateString(),
                'bannerImage' => $gold->banner_image ? url('storage/' . $gold->banner_image) : null,
            ] : null,
            'silver' => $silver ? [
                'silverPriceUpdate' => $silver->price,
                'updateDate' => Carbon::parse($silver->update_date)->toDateString(),
                'bannerImage' => $silver->banner_image ? url('storage/' . $silver->banner_image) : null,
            ] : null,
        ];

        return response()->json($data);
    }

    public function updatePrices(Request $request)
    {
        $request->validate([
            'gold_price' => 'required|numeric',
            'silver_price' => 'required|numeric',
            'banner_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        $bannerImagePath = null;
        if ($request->hasFile('banner_image')) {
            // تخزين الصورة في التخزين العام وإنشاء الرابط
            $bannerImagePath = $request->file('banner_image')->store('banners', 'public');
        }

        $gold = MetalPrice::create([
            'metal_type' => 'gold',
            'price' => $request->gold_price,
            'update_date' => now(),
            'banner_image' => $bannerImagePath,
        ]);

        $silver = MetalPrice::create([
            'metal_type' => 'silver',
            'price' => $request->silver_price,
            'update_date' => now(),
            'banner_image' => $bannerImagePath,
        ]);

        return response()->json([
            'message' => 'Prices updated successfully',
            'gold' => [
                'price' => $gold->price,
                'updateDate' => $gold->update_date,
                'bannerImage' => $gold->banner_image ? url('storage/' . $gold->banner_image) : null,
            ],
            'silver' => [
                'price' => $silver->price,
                'updateDate' => $silver->update_date,
                'bannerImage' => $silver->banner_image ? url('storage/' . $silver->banner_image) : null,
            ],
        ], 201);
    }
    

    public function fetchAndUpdatePrices()
    {
        try {
            // إرسال طلب إلى الموقع الخارجي للحصول على الأسعار
            $response = Http::get('https://saudigoldprice.com/silverprice/');
            
            // التحقق من نجاح الطلب
            if ($response->ok()) {
                $htmlContent = $response->body();

                // استخدم مكتبة مثل DomCrawler أو Regex لتحليل المحتوى
                $goldPrice = $this->extractPrice($htmlContent, 'gold');
                $silverPrice = $this->extractPrice($htmlContent, 'silver');

                if ($goldPrice && $silverPrice) {
                    // تحديث الأسعار في قاعدة البيانات
                    MetalPrice::create([
                        'metal_type' => 'gold',
                        'price' => $goldPrice,
                        'update_date' => now(),
                    ]);

                    MetalPrice::create([
                        'metal_type' => 'silver',
                        'price' => $silverPrice,
                        'update_date' => now(),
                    ]);

                    return response()->json([
                        'message' => 'Prices fetched and updated successfully',
                        'gold_price' => $goldPrice,
                        'silver_price' => $silverPrice,
                    ]);
                } else {
                    return response()->json(['error' => 'Failed to extract prices'], 500);
                }
            } else {
                return response()->json(['error' => 'Failed to fetch data from the external website'], 500);
            }
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    private function extractPrice($htmlContent, $metalType)
    {
        // استخدم تحليل HTML بسيط باستخدام Regex لاستخراج السعر
        if ($metalType === 'gold') {
            preg_match('/<td>الذهب.*?<td>(\d+\.?\d*)<\/td>/', $htmlContent, $matches);
        } elseif ($metalType === 'silver') {
            preg_match('/<td>الفضة.*?<td>(\d+\.?\d*)<\/td>/', $htmlContent, $matches);
        }

        return $matches[1] ?? null; // إذا تم العثور على السعر، يتم إرجاعه
    }


    
    
    
    
}
