<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Models\RecurringDonation;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ProcessRecurringDonationsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

  public function handle()
{
    try {
        Log::info("بدء معالجة التبرعات المتكررة.");

        // احصل على التبرعات المستحقة اليوم
        $donations = RecurringDonation::whereDate('next_donation_date', now()->toDateString())->get();
        
        Log::info("تم العثور على " . $donations->count() . " تبرعات اليوم.");

        if ($donations->isEmpty()) {
            Log::info("لا توجد تبرعات متكررة لمعالجتها اليوم.");
            return; // إذا لم توجد تبرعات، اخرج من الدالة
        }

        $now = Carbon::now(); // الحصول على الوقت الحالي

        foreach ($donations as $donation) {
            Log::info("معالجة التبرع ID: {$donation->id}");

            // تحويل last_donated_at إلى كائن Carbon
            $lastDonatedAt = Carbon::parse($donation->last_donated_at);

            // تحقق من توقيت التبرع
            $shouldDonate = false; // تعيين قيمة البداية

            // تحقق من تكرار التبرع
            if ($donation->frequency == 'daily' && $lastDonatedAt->diffInDays($now) >= 1) {
                $shouldDonate = true;
            } elseif ($donation->frequency == 'weekly' && $lastDonatedAt->diffInWeeks($now) >= 1) {
                $shouldDonate = true;
            } elseif ($donation->frequency == 'monthly' && $lastDonatedAt->diffInMonths($now) >= 1) {
                $shouldDonate = true;
            } elseif ($donation->frequency == 'minutely' && $lastDonatedAt->diffInMinutes($now) >= 1) {
                $shouldDonate = true;
            }

            // إذا كانت التبرعات يجب أن تتم
            if ($shouldDonate) {
                // إرسال طلب الدفع
                Log::info("إرسال طلب دفع لـ ID: {$donation->id}");
                
                $response = Http::withBasicAuth(env('MOYASAR_API_KEY'), '')
                    ->post('https://api.moyasar.com/v1/payments', [
                        "amount" => $donation->amount * 100,
                        "currency" => $donation->currency,
                        "source" => [
                            "type" => "token",
                            "token" => $donation->token,
                            "3ds" => false,
                        ],
                    ]);

                if ($response->successful()) {
                    // تحديث تاريخ التبرع القادم
                    switch ($donation->frequency) {
                        case 'daily':
                            $donation->next_donation_date = $donation->next_donation_date->addDay();
                            break;
                        case 'weekly':
                            $donation->next_donation_date = $donation->next_donation_date->addWeek();
                            break;
                        case 'monthly':
                            $donation->next_donation_date = $donation->next_donation_date->addMonth();
                            break;
                        case 'minutely':
                            $donation->next_donation_date = $donation->next_donation_date->addMinute();
                            break;
                    }

                    $donation->last_donated_at = now(); // تعيين تاريخ آخر تبرع
                    $donation->save();
                    
                    Log::info("تمت معالجة التبرع بنجاح لـ ID: {$donation->id}");
                } else {
                    Log::error("فشل التبرع لـ ID: {$donation->id}. الاستجابة: " . $response->body());
                }
            } else {
                Log::info("لا حاجة للتبرع لـ ID: {$donation->id} في الوقت الحالي.");
            }
        }

        Log::info("انتهت معالجة التبرعات المتكررة.");
    } catch (\Exception $e) {
        Log::error("خطأ في معالجة التبرعات المتكررة: " . $e->getMessage());
    }
}


    public function __construct()
    {
        //
    }
}
