<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use App\Models\Category;

use Illuminate\Support\Facades\Log;

class CategoryController extends Controller
{
    public function index()
    {
        $categories = Category::all();

        foreach ($categories as $category) {
            $category->image_url = $category->image ? url('storage/' . $category->image) : null;
            $category->banner_image_url = $category->banner_image ? url('storage/' . $category->banner_image) : null;
        }

        return response()->json($categories);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string',
            'is_visible' => 'nullable|boolean',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'banner_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('categories', 'public');
            $validated['image'] = $imagePath;
        }

        if ($request->hasFile('banner_image')) {
            $bannerImagePath = $request->file('banner_image')->store('categories/banners', 'public');
            $validated['banner_image'] = $bannerImagePath;
        }

        $category = Category::create($validated);
        $category->image_url = $category->image ? url('storage/' . $category->image) : null;
        $category->banner_image_url = $category->banner_image ? url('storage/' . $category->banner_image) : null;

        return response()->json($category, 201);
    }


public function update(Request $request, $id)
{
    try {
        $request->validate([
            'name' => 'nullable|string',
            'is_visible' => 'nullable|boolean',
            'image' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
            'banner_image' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
        ]);

        $category = Category::find($id);

        if (!$category) {
            return response()->json(['error' => 'Category not found.'], 404);
        }

        $category->name = $request->input('name', $category->name);
        $category->is_visible = $request->input('is_visible', $category->is_visible);

        if ($request->hasFile('image')) {
            if ($category->image) {
                Storage::disk('public')->delete($category->image);
            }
            $imagePath = $request->file('image')->store('categories', 'public');
            $category->image = $imagePath;
        }

        if ($request->hasFile('banner_image')) {
            if ($category->banner_image) {
                Storage::disk('public')->delete($category->banner_image);
            }
            $bannerImagePath = $request->file('banner_image')->store('categories/banners', 'public');
            $category->banner_image = $bannerImagePath;
        }

        $category->save();

        return response()->json([
            'success' => true,
            'message' => 'Category updated successfully.',
            'data' => $category
        ], 200);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'An error occurred while updating the category.',
            'error' => $e->getMessage()
        ], 500);
    }
}




    public function destroy($id)
    {
        $category = Category::findOrFail($id);

        if ($category->image) {
            Storage::disk('public')->delete($category->image);
        }

        if ($category->banner_image) {
            Storage::disk('public')->delete($category->banner_image);
        }

        $category->projects()->delete();

        $category->delete();
        return response()->json(['message' => 'تم حذف الفئة بنجاح'], 200);
    }
}
