<?php

namespace App\Http\Controllers;

use App\Models\Page;
use Illuminate\Http\Request;

class PageController extends Controller
{
    // عرض جميع الصفحات
public function index()
{
    $pages = Page::all()->map(function ($page) {
        return [
            'id' => $page->id,
            'title' => $page->title,
            'page' => $page->page,
            'content' => $page->content,
            'project_id' => $page->project_id,
            'is_visible' => $page->is_visible, 
            'created_at' => $page->created_at,
            'updated_at' => $page->updated_at,
        ];
    });

    return response()->json($pages, 200);
}



    // إنشاء صفحة جديدة
 public function store(Request $request)
{
    // التحقق من صحة البيانات
    $validatedData = $request->validate([
        'title' => 'required|string|max:255',
        'page' => 'required|string|max:255',
        'content' => 'required|string',
        'project_id' => 'nullable|string',
        'media' => 'nullable|file|mimes:jpeg,png,jpg,gif,mp4|max:20480',
        'is_visible' => 'sometimes|boolean', // التحقق من حقل الظهور
    ], [
        // رسائل الخطأ المخصصة
        'title.required' => 'يجب إدخال عنوان الصفحة.',
        'title.string' => 'عنوان الصفحة يجب أن يكون نصاً.',
        'page.required' => 'يجب إدخال اسم الصفحة.',
        'page.string' => 'اسم الصفحة يجب أن يكون نصاً.',
        'content.required' => 'يجب إدخال المحتوى.',
        'media.mimes' => 'يجب أن يكون الملف من نوع: jpeg, png, jpg, gif, أو mp4.',
        'media.max' => 'حجم الملف يجب ألا يتجاوز 200 ميجابايت.',
        'project_id.string' => 'معرف المشروع يجب أن يكون نصًا.',
        'is_visible.boolean' => 'حقل الظهور يجب أن يكون صحيحاً أو خطأ.',
    ]);

    // حذف جميع الصفحات القديمة
    Page::truncate(); // حذف جميع السجلات من جدول Page

    // إذا كان هناك ملف تم رفعه
    if ($request->hasFile('media')) {
        $filePath = $request->file('media')->store('media', 'public');
        $validatedData['content'] .= "\nMedia URL: " . url('storage/' . $filePath);
    }

    // إذا تم إرسال قيمة لحقل is_visible، أضفها للبيانات
    if ($request->has('is_visible')) {
        $validatedData['is_visible'] = $request->input('is_visible');
    }

    // إنشاء الصفحة الجديدة
    $page = Page::create($validatedData);

    return response()->json($page, 201);
}


    // عرض صفحة معينة
    public function show($id)
    {
        $page = Page::findOrFail($id);

        return response()->json($page);
    }

    // تحديث صفحة معينة
public function update(Request $request, $id)
{
    // التحقق من صحة البيانات
    $validated = $request->validate([
        'title' => 'sometimes|string|max:255',
        'page' => 'sometimes|string|max:255',
        'content' => 'sometimes|json',
        'project_id' => 'nullable|string',
        'is_visible' => 'sometimes|boolean', // التحقق من حقل is_visible
    ], [
        // رسائل الخطأ المخصصة
        'title.string' => 'عنوان الصفحة يجب أن يكون نصاً.',
        'page.string' => 'اسم الصفحة يجب أن يكون نصاً.',
        'content.json' => 'يجب أن يكون المحتوى بتنسيق JSON.',
        'project_id.string' => 'معرف المشروع يجب أن يكون نصًا.',
        'is_visible.boolean' => 'حقل الظهور يجب أن يكون صحيحاً أو خطأ.',
    ]);

    // إيجاد الصفحة وتحديثها
    $page = Page::findOrFail($id);
    $page->update($validated);

    return response()->json([
        'message' => 'تم تحديث الصفحة بنجاح.',
        'page' => $page,
    ]);
}



    // حذف صفحة معينة
    public function destroy($id)
    {
        $page = Page::findOrFail($id);
        $page->delete();

        return response()->json(null, 204);
    }
}
