<?php

namespace App\Http\Controllers;

use App\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class ProjectController extends Controller
{
    public function index()
    {
        $projects = Project::where(function ($query) {
            $query->whereNull('end_date')
                  ->orWhere('end_date', '>', now());
        })
        ->whereHas('category', function ($query) {
            $query->where('is_visible', true);
        })
        ->get();

        $formattedProjects = $projects->map(function ($project) {
            // زيادة عدد الزيارات
            $project->increment('visits');

            return [
                'id' => $project->id,
                'category_id' => $project->category_id,
                'category' => $project->category->name,
                'title' => $project->title,
                'imgSrc' => $project->imgSrc,
                'description' => $project->description,
                'beneficiaries_count' => $project->beneficiaries_count,
                'target_count' => $project->target_count,
                'start_date' => $project->start_date,
                'end_date' => $project->end_date,
                'is_visible' => $project->is_visible,
                'totalMoney' => $project->totalMoney,
                'gained' => $project->gained,
                'percentage' => $project->show_percentage ? $project->percentage : null,
                'show_percentage' => $project->show_percentage,
                'show_gained' => $project->show_gained,
                'show_totalMoney' => $project->show_totalMoney,
                'created_at' => $project->created_at,
                'updated_at' => $project->updated_at,
                'visits' => $project->visits,
                'participantName' => $project->participantName,
                'amount' => $project->amount,
                'deduction_period' => $project->deduction_period,
                'deduction_end_date' => $project->deduction_end_date,
                'seo_keywords' => $project->seo_keywords,
                'seo_description' => $project->seo_description,
                'related_project_ids' => $project->related_project_ids,
                'showSahm' => $project->showSahm,
                'sahm_cart' => $project->sahm_cart, // تأكد من وجود هذه الحقول
                'sahm_quick' => $project->sahm_quick, // تأكد من وجود هذه الحقول
                'specified_amount' => $project->specified_amount,  
                'sort' => $project->sort,
            ];
        });

        return response()->json($formattedProjects);
    }

public function store(Request $request)
{
    try {
        // التحقق من صحة البيانات
        $validated = $request->validate([
            'category_id' => 'required|exists:categories,id',
            'title' => 'required|string',
            'totalMoney' => 'required|numeric',
            'gained' => 'nullable|numeric',
            'img' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'sahm_cart' => 'nullable|array',
            'sahm_quick' => 'nullable|array',
            'description' => 'nullable|string',
            'percentage' => 'nullable|numeric',
            'show_percentage' => 'nullable|boolean',
            'show_gained' => 'nullable|boolean',
            'show_totalMoney' => 'nullable|boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date',
            'is_visible' => 'nullable|boolean',
            'participantName' => 'nullable|string',
            'amount' => 'nullable|numeric',
            'deduction_period' => 'nullable|string',
            'deduction_end_date' => 'nullable|date',
            'seo_keywords' => 'nullable|string',
            'seo_description' => 'nullable|string',
            'related_project_ids' => 'nullable|string',
            'showSahm' => 'nullable|string|in:cart,quick,both',
            'cart_donation_type' => 'nullable|string',
            'quick_donation_type' => 'nullable|string',
            'cart_deduction_date' => 'nullable|date',
            'quick_deduction_date' => 'nullable|date',
            'specified_amount' => 'nullable|numeric',
            'sort' => 'nullable|integer|min:0', // الحقل الجديد
        ]);

        // التعامل مع الصورة
        if ($request->hasFile('img')) {
            $imgPath = $request->file('img')->store('images', 'public');
            $validated['imgSrc'] = Storage::url($imgPath);
        }

        if (isset($validated['related_project_ids'])) {
            $validated['related_project_ids'] = implode(',', explode(',', $validated['related_project_ids']));
        }

        // تحويل القيم إلى JSON فقط إذا لم تكن بالفعل JSON
        $validated['sahm_cart'] = is_array($request->input('sahm_cart', [])) ? json_encode($request->input('sahm_cart', [])) : $request->input('sahm_cart', '[]');
        $validated['sahm_quick'] = is_array($request->input('sahm_quick', [])) ? json_encode($request->input('sahm_quick', [])) : $request->input('sahm_quick', '[]');

        // إعداد البيانات
        $projectData = array_merge($validated, [
            'show_percentage' => $request->boolean('show_percentage', true),
            'show_gained' => $request->boolean('show_gained', true),
            'show_totalMoney' => $request->boolean('show_totalMoney', true),
            'is_visible' => $request->boolean('is_visible', true),
            'sort' => $request->input('sort', 0), // القيمة الافتراضية 0
        ]);

        // إنشاء المشروع
        $project = Project::create($projectData);
        $project->calculatePercentageValues();

        return response()->json($project, 201);

    } catch (\Illuminate\Validation\ValidationException $e) {
        return response()->json([
            'error' => 'بيانات غير صحيحة',
            'details' => $e->errors()
        ], 422);
    } catch (\Exception $e) {
        Log::error('Exception: ' . $e->getMessage());
        Log::error('Trace: ' . $e->getTraceAsString());

        return response()->json([
            'error' => 'حدث خطأ غير متوقع',
            'details' => [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]
        ], 500);
    }
}



public function update(Request $request, $id)
{
    try {
        // التحقق من صحة البيانات
        $validated = $request->validate([
            'category_id' => 'nullable|exists:categories,id',
            'title' => 'nullable|string',
            'totalMoney' => 'nullable|numeric',
            'gained' => 'nullable|numeric',
            'img' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'sahm_cart' => 'nullable|array',
            'sahm_quick' => 'nullable|array',
            'description' => 'nullable|string',
            'percentage' => 'nullable|numeric',
            'show_percentage' => 'nullable|boolean',
            'show_gained' => 'nullable|boolean',
            'show_totalMoney' => 'nullable|boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date',
            'is_visible' => 'nullable|boolean',
            'participantName' => 'nullable|string',
            'amount' => 'nullable|numeric',
            'deduction_period' => 'nullable|string',
            'deduction_end_date' => 'nullable|date',
            'seo_keywords' => 'nullable|string',
            'seo_description' => 'nullable|string',
            'related_project_ids' => 'nullable|string',
            'showSahm' => 'nullable|string|in:cart,quick,both',
            'cart_donation_type' => 'nullable|string',
            'quick_donation_type' => 'nullable|string',
            'cart_deduction_date' => 'nullable|date',
            'quick_deduction_date' => 'nullable|date',
            'specified_amount' => 'nullable|numeric',
            'sort' => 'nullable|integer|min:0', // الحقل الجديد
        ]);

        // العثور على المشروع
        $project = Project::findOrFail($id);

        // التعامل مع الصورة
        if ($request->hasFile('img')) {
            $imgPath = $request->file('img')->store('images', 'public');
            $validated['imgSrc'] = Storage::url($imgPath);
        }

        if (isset($validated['related_project_ids'])) {
            $validated['related_project_ids'] = implode(',', explode(',', $validated['related_project_ids']));
        }

        // تحويل الحقول sahm_cart و sahm_quick إلى JSON مشفر
    // تحويل الحقول sahm_cart و sahm_quick إلى JSON مشفر أو مسحها إذا كانت فارغة
$validated['sahm_cart'] = $request->has('sahm_cart') && is_array($request->input('sahm_cart')) 
    ? json_encode($request->input('sahm_cart')) 
    : null; // مسح القيمة إذا كانت الحقول فارغة أو غير موجودة

$validated['sahm_quick'] = $request->has('sahm_quick') && is_array($request->input('sahm_quick')) 
    ? json_encode($request->input('sahm_quick')) 
    : null; // مسح القيمة إذا كانت الحقول فارغة أو غير موجودة


        // إعداد البيانات
        $projectData = array_merge($validated, [
            'show_percentage' => $request->has('show_percentage') ? $request->boolean('show_percentage') : $project->show_percentage,
            'show_gained' => $request->has('show_gained') ? $request->boolean('show_gained') : $project->show_gained,
            'show_totalMoney' => $request->has('show_totalMoney') ? $request->boolean('show_totalMoney') : $project->show_totalMoney,
            'is_visible' => $request->has('is_visible') ? $request->boolean('is_visible') : $project->is_visible,
            'sort' => $request->has('sort') ? $request->input('sort') : $project->sort, // القيمة الافتراضية من المخزن
        ]);

        // تحديث المشروع
        $project->update($projectData);
        $project->calculatePercentageValues();

        return response()->json($project, 200);

    } catch (\Illuminate\Validation\ValidationException $e) {
        return response()->json([
            'error' => 'بيانات غير صحيحة',
            'details' => $e->errors()
        ], 422);
    } catch (\Exception $e) {
        Log::error('Exception: ' . $e->getMessage());
        Log::error('Trace: ' . $e->getTraceAsString());

        return response()->json([
            'error' => 'حدث خطأ غير متوقع',
            'details' => [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]
        ], 500);
    }
}






    public function show($id)
    {
        $project = Project::findOrFail($id);
        $project->increment('visits');
        
        return response()->json($project);
    }

    public function delete($id)
    {
        $project = Project::findOrFail($id);
        $project->delete();
        return response()->json(['message' => 'تم حذف المشروع بنجاح'], 200);
    }

    public function toggleVisibility($id)
    {
        $project = Project::findOrFail($id);
        $project->is_visible = !$project->is_visible; // تغيير حالة العرض
        $project->save();

        return response()->json($project, 200);
    }
}
