<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\QrCode;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use SimpleSoftwareIO\QrCode\Facades\QrCode as QrCodeGenerator;

class QrCodeController extends Controller
{
    /**
     * Display a listing of the resource.
     */
public function index()
{
    try {
        $qrCodes = QrCode::all();

        // إضافة المسار الكامل للصورة
        $qrCodes->transform(function ($qrCode) {
            // إضافة المسار الكامل للـ QR Code
            $qrCode->qr_code_url = url('storage/' . $qrCode->qr_code);
            return $qrCode;
        });

        return response()->json($qrCodes);
    } catch (\Exception $e) {
        return response()->json(['error' => 'An error occurred while retrieving QR codes.'], 500);
    }
}


    /**
     * Store a newly created resource in storage.
     */
public function store(Request $request)
{
    $request->validate([
        'title' => 'required|string',
        'url' => 'required|url',
    ]);

    // توليد QR Code كصورة
    $qrCodeImage = QrCodeGenerator::format('png')->size(300)->generate($request->url);

    // حفظ الصورة في التخزين العام
    $fileName = uniqid() . '.png';
    $filePath = 'qrcodes/' . $fileName;
    \Storage::disk('public')->put($filePath, $qrCodeImage);

    // إنشاء السجل في قاعدة البيانات
    $qrCode = QrCode::create([
        'title' => $request->title,
        'url' => $request->url,
        'qr_code' => $filePath, // مسار الصورة المخزن
    ]);

    // إرجاع الرد مع رابط الصورة
    return response()->json([
        'id' => $qrCode->id,
        'title' => $qrCode->title,
        'url' => $qrCode->url,
        'qr_code_url' => url('storage/' . $filePath),
    ], 201);
}


    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $qrCode = QrCode::findOrFail($id);
            $qrCode->update($request->all());
            return response()->json($qrCode);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'QR code not found.'], 404);
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred while updating QR code.'], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $qrCode = QrCode::findOrFail($id);
            $qrCode->delete();
            return response()->json(['message' => 'Deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'QR code not found.'], 404);
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred while deleting QR code.'], 500);
        }
    }
}
