<?php

namespace App\Http\Controllers;

use App\Models\Target;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TargetController extends Controller
{
public function index()
{
    $targets = Target::all();

    // تنسيق البيانات بالشكل المطلوب
    $formattedTargets = $targets->map(function ($target) {
        return [
            'target_name' => $target->target_name,
            'target_amount' => $target->target_amount,
            'date' => $target->date,
            'status' => $target->status,
            'target_settings' => json_decode($target->target_settings) ?? [], // تحويل JSON إلى مصفوفة
            'updated_at' => $target->updated_at,
            'created_at' => $target->created_at,
            'id' => $target->id,
            'quick_donate' => json_decode($target->quick_donate) ?? [], // تحويل JSON إلى مصفوفة
            'projects' => json_decode($target->projects) ?? [], // تحويل JSON إلى مصفوفة
            'gifts' => json_decode($target->gifts) ?? [], // تحويل JSON إلى مصفوفة
        ];
    });

    return response()->json(['targets' => $formattedTargets]);
}

    
    private function getQuickDonates(Target $target)
{
    return $target->quickDonates; // افترض أن لديك علاقة في نموذج Target
}

private function getProjects(Target $target)
{
    return $target->projects; // افترض أن لديك علاقة في نموذج Target
}

private function getGifts(Target $target)
{
    return $target->gifts; // افترض أن لديك علاقة في نموذج Target
}

    
    public function show($id)
{
    $target = Target::findOrFail($id);
    return response()->json([
        'target_name' => $target->target_name,
        'target_amount' => $target->target_amount,
        'status' => $target->status,
        'quick_donate' => json_decode($target->quick_donate) ?? [], // تحويل JSON إلى مصفوفة
        'projects' => json_decode($target->projects) ?? [], // تحويل JSON إلى مصفوفة
        'gifts' => json_decode($target->gifts) ?? [], // تحويل JSON إلى مصفوفة
    ]);
}


    public function store(Request $request)
    {
        $request->validate([
            'target_name' => 'required|string|max:255',
            'target_amount' => 'required|numeric',
            'date' => 'nullable|date',
            'status' => 'required|in:active,inactive',
            'target_settings' => 'nullable|json',
            'quick_donate' => 'nullable|array',
            'projects' => 'nullable|array',
            'gifts' => 'nullable|array',
        ]);

        DB::transaction(function () use ($request) {
            $target = Target::create([
                'target_name' => $request->target_name,
                'target_amount' => $request->target_amount,
                'date' => $request->date,
                'status' => $request->status,
                'target_settings' => $request->target_settings,
            ]);

            if ($request->filled('quick_donate')) {
                $this->storeQuickDonates($target, $request->quick_donate);
            }

            if ($request->filled('projects')) {
                $this->storeProjects($target, $request->projects);
            }

            if ($request->filled('gifts')) {
                $this->storeGifts($target, $request->gifts);
            }
        });

        return response()->json(['message' => 'Target created successfully'], 201);
    }

    private function storeQuickDonates($target, $quickDonates)
    {
        // منطق لحفظ بيانات التبرعات السريعة
    }

    private function storeProjects($target, $projects)
    {
        // منطق لحفظ بيانات المشاريع
    }

    private function storeGifts($target, $gifts)
    {
        // منطق لحفظ بيانات الهدايا
    }

public function update(Request $request, $id)
{
    $target = Target::findOrFail($id);

    $request->validate([
        'target_name' => 'sometimes|string|max:255',
        'target_amount' => 'sometimes|numeric',
        'status' => 'sometimes|in:active,inactive',
        'target_settings' => 'sometimes|nullable|json',
        'quick_donate' => 'nullable|array',
        'projects' => 'nullable|array',
        'gifts' => 'nullable|array',
        'remove_quick_donate' => 'nullable|array',
        'remove_projects' => 'nullable|array',
        'remove_gifts' => 'nullable|array',
    ]);

    DB::transaction(function () use ($request, $target) {
        // تحديث الحقول المطلوبة
        $target->update($request->only(['target_name', 'target_amount', 'status', 'target_settings']));

        // تحديث بيانات التبرعات السريعة
        if ($request->filled('quick_donate')) {
            $target->quick_donate = json_encode($request->quick_donate);
        }

        // تحديث بيانات المشاريع
        if ($request->filled('projects')) {
            $target->projects = json_encode($request->projects);
        }

        // تحديث بيانات الهدايا
        if ($request->filled('gifts')) {
            $target->gifts = json_encode($request->gifts);
        }

        // إزالة العناصر المحددة
        $this->removeItems($target, 'quick_donate', $request->remove_quick_donate);
        $this->removeItems($target, 'projects', $request->remove_projects);
        $this->removeItems($target, 'gifts', $request->remove_gifts);

        // حفظ التغييرات
        $target->save();
    });

    // إرجاع الاستجابة بالشكل المطلوب بعد التحديث
    return response()->json([
        'target_name' => $target->target_name,
        'target_amount' => $target->target_amount,
        'status' => $target->status,
        'quick_donate' => json_decode($target->quick_donate, true) ?? [], // تحويل JSON إلى مصفوفة
        'projects' => json_decode($target->projects, true) ?? [], // تحويل JSON إلى مصفوفة
        'gifts' => json_decode($target->gifts, true) ?? [], // تحويل JSON إلى مصفوفة
    ]);
}

protected function removeItems($target, $field, $itemsToRemove)
{
    if ($itemsToRemove && is_array($itemsToRemove)) {
        // فك تشفير البيانات الحالية
        $currentItems = json_decode($target->$field, true) ?? [];

        // إزالة العناصر المحددة
        foreach ($itemsToRemove as $itemToRemove) {
            $currentItems = array_filter($currentItems, function ($item) use ($itemToRemove) {
                return $item['siteName'] !== $itemToRemove['siteName'];
            });
        }

        // تحديث القيمة في قاعدة البيانات
        $target->$field = json_encode(array_values($currentItems));
    }
}



    public function destroy($id)
    {
        $target = Target::findOrFail($id);
        $target->delete();

        return response()->json(['message' => 'تم حذف الهدف بنجاح']);
    }


}

