<?php

namespace App\Http\Controllers;

use App\Models\WaqfCertificate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class WaqfCertificateController extends Controller
{
    public function index()
    {
        $waqfCertificates = WaqfCertificate::all()->map(function ($certificate) {
            $certificate->banner_image_url = Storage::url($certificate->banner_image);
            return $certificate;
        });

        return response()->json($waqfCertificates);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'amount' => 'required|numeric|min:1000',
            'issued_by' => 'required|string|max:255',
            'banner_image' => 'required|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $bannerImage = $request->file('banner_image')->store('banners', 'public');

        $waqfCertificate = WaqfCertificate::create([
            'name' => $request->name,
            'amount' => $request->amount,
            'issued_by' => $request->issued_by,
            'banner_image' => $bannerImage,
        ]);

        $waqfCertificate->banner_image_url = Storage::url($bannerImage);

        return response()->json($waqfCertificate, 201);
    }

    public function show($id)
    {
        $waqfCertificate = WaqfCertificate::findOrFail($id);
        $waqfCertificate->banner_image_url = Storage::url($waqfCertificate->banner_image);
        return response()->json($waqfCertificate);
    }

    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|string|max:255',
            'amount' => 'sometimes|numeric|min:1000',
            'issued_by' => 'sometimes|string|max:255',
            'banner_image' => 'sometimes|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $waqfCertificate = WaqfCertificate::findOrFail($id);

        if ($request->hasFile('banner_image')) {
            // Delete old image
            Storage::disk('public')->delete($waqfCertificate->banner_image);
            // Store new image
            $waqfCertificate->banner_image = $request->file('banner_image')->store('banners', 'public');
        }

        $waqfCertificate->update($request->only(['name', 'amount', 'issued_by']));

        $waqfCertificate->banner_image_url = Storage::url($waqfCertificate->banner_image);

        return response()->json($waqfCertificate);
    }

    public function destroy($id)
    {
        $waqfCertificate = WaqfCertificate::findOrFail($id);
        // Delete image
        Storage::disk('public')->delete($waqfCertificate->banner_image);
        $waqfCertificate->delete();
        return response()->json(['message' => 'Deleted successfully']);
    }
}
